// Copyright ® 2025 OneSpan North America, Inc. All rights reserved. 

 
/////////////////////////////////////////////////////////////////////////////
//
//
// This file is example source code. It is provided for your information and
// assistance. See your licence agreement for details and the terms and
// conditions of the licence which governs the use of the source code. By using
// such source code you will be accepting these terms and conditions. If you do
// not wish to accept these terms and conditions, DO NOT OPEN THE FILE OR USE
// THE SOURCE CODE.
//
// Note that there is NO WARRANTY.
//
//////////////////////////////////////////////////////////////////////////////


import UIKit
import AudioToolbox

class NotificationViewController: UIViewController {
    // MARK: - IB Outlets
    @IBOutlet private weak var notificationIdentifierText: UITextView!
    @IBOutlet private weak var notificationContentText: UITextView!
    @IBOutlet private weak var notificationCustomAttributes: UITextView!
    @IBOutlet private weak var errorLabel: UILabel!
    
    // MARK: - Private properties
    private let notificationSoundID: SystemSoundID = 1007
    private weak var appDelegate: AppDelegate? = {
        UIApplication.shared.delegate as? AppDelegate
    }()
    
    // MARK: - App Lifecycle
    override func viewDidLoad() {
        super.viewDidLoad()
    }

    override func viewWillAppear(_ animated: Bool) {
        super.viewWillAppear(animated)
        
        // subscribe to notification data updates
        NotificationCenter.default.addObserver(
            self,
            selector: #selector(dataUpdated),
            name: Notification.Name.notificationDataUpdated,
            object: nil
        )
        
        // subscribe to notification when application notification received
        NotificationCenter.default.addObserver(
            self,
            selector: #selector(notificationReceived),
            name: Notification.Name.notificationReceived,
            object: nil
        )
    }
    
    override func viewWillDisappear(_ animated: Bool) {
        super.viewWillDisappear(animated)
        
        NotificationCenter.default.removeObserver(self)
    }
    
    // MARK: - Private functions
    
    /// Called when the notification data are modified.
    /// - Parameter notification: notification
    @objc private func dataUpdated(notification: NSNotification) {
        // simply update UI
        updateUI()
    }
    
    /// Called when a notification is received.
    /// - Parameter notification: notification object passed by NotificationCenter
    @objc private func notificationReceived(notification: NSNotification) {
        // simply update UI
        displayNotification()
    }

    /// Update the UI fields with the notification data provided by the AppDelegate.
    private func updateUI() {
        errorLabel.text = appDelegate?.errorText
        notificationIdentifierText.text = appDelegate?.notificationIdentifierText
        
        // Directly copy identifier to clipboard if present
        if let notificationIdentifier = appDelegate?.notificationIdentifierText, !notificationIdentifier.isEmpty {
            UIPasteboard.general.string = notificationIdentifier
        }
    }
    
    /// Display the notification on the screen.
    private func displayNotification() {
        notificationContentText.text = appDelegate?.oneSpanNotification?.oneSpanContent
        let customAttributes = appDelegate?.oneSpanNotification?.customAttributes
        if let customAttributes = customAttributes, !customAttributes.isEmpty {
            var notificationCustomAttributesText = "Custom attributes :\n"
            for (attributeKey, attributeValue) in customAttributes {
                notificationCustomAttributesText += attributeKey + " - " + attributeValue + "\n"
            }
            notificationCustomAttributes.text = notificationCustomAttributesText
        } else {
            notificationCustomAttributes.text = ""
        }
        
        if appDelegate?.notificationFromBackground == false {
            // Added sound for notifications in foreground
            AudioServicesPlayAlertSound(notificationSoundID)
            
            // If the notification has been received when application was in foreground,
            // we display the title and the subject of the notification
            let title = appDelegate?.oneSpanNotification?.title
            let message = appDelegate?.oneSpanNotification?.subject
            
            let alert = UIAlertController(title: title, message: message, preferredStyle: .alert)
            let okAction = UIAlertAction(title: "OK", style: .default)
            alert.addAction(okAction)
            self.present(alert, animated: true, completion: nil)
        }
        
        // reset the flag for next notification
        appDelegate?.notificationFromBackground = false
    }
}
